using System;
using System.Diagnostics;
using Zcu.Mve.Core;
using Zcu.Mve.Numerics;

namespace Zcu.Mve.Numerics
{
	/// <summary>
	/// Source of n-dimensional vector.
	/// </summary>
	[ModuleInfo("Milan Frank", "Source of n-dimensional vector.", Assemblied = "2004-09-4", IconName="VectorSource.ico")]
	[PortInfo("output", "n-dimensional vector")]
	public class VectorSource : Zcu.Mve.Core.Module
	{
		internal IVector vect;

		/// <summary>
		/// Creates ports.
		/// </summary>
		public VectorSource()
		{
			this.AddOutPort("output", typeof (VectorND));
			vect = new VectorND(0.0);
		}
	
		/// <summary>
		/// Executes the VectorSource functionality.
		/// </summary>
		public override void Execute()
		{
			this.SetOutput("output", vect);
		}
  
		/// <summary>
		/// Invoke model setup.
		/// </summary>
		/// <returns>VectorSource setup dialog.</returns>
		public override ModuleSetup InvokeSetup()
		{
			return new VectorSourceSetup(this);
		}
  
		/// <summary>
		/// Read the configuration data from XML file.
		/// </summary>
		/// <param name="config">XmlElement containing the configuration data.</param>
		public override void ReadConfig(System.Xml.XmlElement config)
		{
			string [] vals = config.InnerText.Trim().Split(' ');

			// config is void
			if (vals.Length == 1 && vals[0].Length < 1)
				return;

			VectorND tmpVect = new VectorND(new double[vals.Length]);

			try 
			{
				int i = 0;
				foreach (string str in vals)
				{
					tmpVect[i] = Double.Parse(str, Globals.nfi);
					i++;
				}
			}
			catch (FormatException ex)
			{
				Debug.WriteLine(ex.Message);
				// zavraneni ukonceni dialogu
				throw new Zcu.Mve.Core.MveException("Incorrect number format in Config file.");
			}    

			switch (config.Attributes["type"].Value)
			{
				case "Vector2D":
				{
					Vect = (Vector2D) tmpVect;
					break;
				}
				case "Vector3D":
				{
					Vect = (Vector3D) tmpVect;
					break;
				}
				case "VectorND":
					Vect = (VectorND) tmpVect;
					break;
			}
		}
  
		/// <summary>
		/// Write the configuration data in XML file.
		/// </summary>
		/// <param name="config">XmlElement containing the configuration data.</param>
		public override void WriteConfig(System.Xml.XmlElement config)
		{
			config.SetAttribute("type", vect.GetType().Name);

			switch (vect.GetType().Name)
			{
				case "Vector2D":
				{
					Vector2D v = (Vector2D)vect;
					config.InnerText = v.X.ToString(Globals.nfi) + " " + v.Y.ToString(Globals.nfi);
					break;
				}
				case "Vector3D":
				{
					Vector3D v = (Vector3D)vect;
					config.InnerText = v.X.ToString(Globals.nfi) + " " + v.Y.ToString(Globals.nfi) +  " " + v.Z.ToString(Globals.nfi);
					break;
				}
				case "VectorND":
					foreach (double d in ((VectorND)vect).Values)
					{
						config.InnerText += d.ToString(Globals.nfi) + " ";
					}
					break;
			}
		}

		/// <summary>
		/// Get/set vector.
		/// </summary>
		internal IVector Vect
		{
			get
			{
				return vect;
			}
			set
			{
				if (vect.GetType() != value.GetType())
				{
					RemoveOutPort("output");
					AddOutPort("output", value.GetType());
				}
				vect = value;
			}
		}
	}
}
