using System;
using System.Drawing;
using System.Collections;
using System.ComponentModel;
using System.Windows.Forms;
using System.Globalization;
using Zcu.Mve.Numerics;

namespace Zcu.Mve.Numerics
{
	/// <summary>
	/// Panel for vector rendering.
	/// </summary>
	public class VectorRendererPanel : System.Windows.Forms.Form
	{
		private System.ComponentModel.IContainer components = null;
		private System.Windows.Forms.StatusBar statusBar1;
		private System.Windows.Forms.StatusBarPanel statusBarPanel1;
		private System.Windows.Forms.MainMenu mainMenu1;
		private System.Windows.Forms.MenuItem menuItem1;

		/// <summary>
		/// Vector data - n-dimensional arrary of vector values.
		/// </summary>
		private double[] data;

		#region Constants and variables for settings of graph properties
		/// <summary>
		/// x-resolution of panel.
		/// </summary>
		private int max_x = 600;
		/// <summary>
		/// y-resolution of panel.
		/// </summary>
		private int max_y = 400;

		/// <summary>
		/// Graph-top window margin spacing.
		/// </summary>
		private const float SPACING_TOP = 30;
		/// <summary>
		/// Graph-bottom window margin spacing.
		/// </summary>
		private const float SPACING_BOTTOM = 50;		// vetn status bar panelu
		/// <summary>
		/// Graph-left window margin spacing.
		/// </summary>
		private const float SPACING_LEFT = 50;
		/// <summary>
		/// Graph-right window margin spacing.
		/// </summary>
		private const float SPACING_RIGHT = 30;

		/// <summary>
		/// Graph width.
		/// </summary>
		private float width;
		/// <summary>
		/// Graph heihgt.
		/// </summary>
		private float height;

		/// <summary>
		/// Axis marker size.
		/// </summary>
		private const float AXIS_MARK_SIZE = 5;

		/// <summary>
		/// Graph marker size.
		/// </summary>
		internal float graph_mark_size = 6;

		/// <summary>
		/// Count of y-segments.
		/// </summary>
		internal int y_segment_count = 10;

		/// <summary>
		/// Max value of vector elements.
		/// </summary>
		private double maxValue;
		/// <summary>
		/// Min value of vector elements.
		/// </summary>
		private double minValue;

		/// <summary>
		/// x-segment size.
		/// </summary>
		private float segmentX;
		/// <summary>
		/// y-segment size.
		/// </summary>
		private float segmentY;

		/// <summary>
		/// Show x grid.
		/// </summary>
		internal bool showXGrid = false;
		/// <summary>
		/// Show y grid.
		/// </summary>
		internal bool showYGrid = false;
		/// <summary>
		/// Show x axes legend.
		/// </summary>
		internal bool showXAxesLegend = false;
		/// <summary>
		/// Show y axes legend.
		/// </summary>
		internal bool showYAxesLegend = false;
		/// <summary>
		/// Show graph markers.
		/// </summary>
		internal bool showGraphMarks = true;
		/// <summary>
		/// Graph line width.
		/// </summary>
		internal int graphLineWidth = 2;
		/// <summary>
		/// Graph marker size.
		/// </summary>
		internal int graphMarkLineWidth = 3;
		/// <summary>
		/// Color of graph line.
		/// </summary>
		internal Color graphLineColor = Color.Blue;
		/// <summary>
		/// Color of markers.
		/// </summary>
		internal Color marksLineColor = Color.Red;
		#endregion

		/// <summary>
		/// Creates and initialize the panel.
		/// </summary>
		/// <param name="data"></param>
		public VectorRendererPanel(IVector data)
		{
			if (data.GetType() == typeof(Vector2D))
				this.data = new double[] {((Vector2D)data).X, ((Vector2D)data).Y};
			else if (data.GetType() == typeof(Vector3D))
				this.data = new double[] {((Vector3D)data).X, ((Vector3D)data).Y, ((Vector3D)data).Z};
			else	// zadan n-rozmerny vektor
				this.data = ((VectorND)data).Values;

			this.minValue = this.FindMinInData(this.data);
			this.maxValue = this.FindMaxInData(this.data);
			//
			// Required for Windows Form Designer support
			//
			InitializeComponent();
			this.ClientSize = new Size(max_x, max_y);
			this.statusBarPanel1.Text = " Renderered data: " + data.ToString();
			this.width  = max_x - VectorRendererPanel.SPACING_LEFT - VectorRendererPanel.SPACING_RIGHT;
			this.height = max_y - VectorRendererPanel.SPACING_TOP - VectorRendererPanel.SPACING_BOTTOM;
			this.segmentX = width  / (float) (this.data.Length - 1);
			this.segmentY = height / (float) y_segment_count;
		} // VectorRendererPanel()

		/// <summary>
		/// Draw the graph.
		/// </summary>
		/// <param name="e">Event.</param>
		protected override void OnPaint(PaintEventArgs e)
		{
			Graphics g = e.Graphics;
			this.RenderVector(g);
			base.OnPaint (e);
		} // OnPaint()

		/// <summary>
		/// Refresh the panel after resize.
		/// </summary>
		/// <param name="e">Event.</param>
		protected override void OnResize(EventArgs e)
		{
			this.Refresh();
			base.OnResize (e);
		} // OnResize()

		/// <summary>
		/// Refresh the panel.
		/// </summary>
		public override void Refresh()
		{
			this.max_x = this.ClientSize.Width;
			this.max_y = this.ClientSize.Height;
			this.width  = max_x - VectorRendererPanel.SPACING_LEFT - VectorRendererPanel.SPACING_RIGHT;
			this.height = max_y - VectorRendererPanel.SPACING_TOP - VectorRendererPanel.SPACING_BOTTOM;
			this.segmentX = width  / (float) (data.Length - 1);
			this.segmentY = height / (float) y_segment_count;
			this.RenderVector(this.CreateGraphics());
			base.Refresh ();
		} // Refresh()

		#region Pomocn metody pro vykreslovn grafu
		/// <summary>
		/// Vraci maximalni prvek vektoru.
		/// </summary>
		/// <param name="data">Vektor hodnot.</param>
		/// <returns>Maximalni prvek vektoru.</returns>
		private double FindMaxInData(double[] data)
		{
			double max = Double.MinValue;
			for (int i = 0; i < data.Length; i++)
				if (data[i] > max)
					max = data[i];
			if (max == this.minValue)
				return max + 1;		// kvuli lepsimu vykresleni grafu
			else return max;
		} // FindMaxInData()

		/// <summary>
		/// Vraci minimalni prvek vektoru.
		/// </summary>
		/// <param name="data">Vektor hodnot.</param>
		/// <returns>Minimalni prvek vektoru.</returns>
		private double FindMinInData(double[] data)
		{
			double min = Double.MaxValue;
			for (int i = 0; i < data.Length; i++)
				if (data[i] < min)
					min = data[i];
			return min;
		} // FindMinInData()

		/// <summary>
		/// Vykresl cely graf vektoru (vcetne os, mrizky a popisku).
		/// Jsou-li vstupni data skalar, je vypsana chybova hlaska a graf se nevykresli.
		/// </summary>
		/// <param name="g">Kreslici povrch.</param>
		private void RenderVector(Graphics g)
		{
			if (this.data.Length > 1)
			{
				// Inicializace pouzitych kreslicich nastroju.
				Pen axisPen = new Pen(new SolidBrush(Color.Black), 2);
				Pen graphPen = new Pen(new SolidBrush(this.graphLineColor), graphLineWidth);
				Font axisFont = new Font(FontFamily.GenericSansSerif, 8, FontStyle.Bold);
				// Vodorovna osa grafu
				DrawXAxis(g, axisPen, axisFont);
				// Vodorovna osa grafu
				DrawYAxis(g, axisPen, axisFont);
				// Ohraniceni grafu - obdelnik okolo.	
				g.DrawRectangle(Pens.Black, VectorRendererPanel.SPACING_LEFT, 
					VectorRendererPanel.SPACING_TOP, 
					max_x - VectorRendererPanel.SPACING_RIGHT - VectorRendererPanel.SPACING_LEFT, 
					max_y - VectorRendererPanel.SPACING_BOTTOM - VectorRendererPanel.SPACING_TOP);
				// Vykresleni vlastniho grafu
				DrawMyGraph(g, graphPen);
			}
			else
			{
				Font warningFont = new Font(FontFamily.GenericMonospace, 10, FontStyle.Bold);
				g.DrawString("           -== ERROR: ==-\n"
					+ "    Input data must be a Vector.\n"
					+ "This is not Vector, but Scalar value.\n"
					+ "  Cannot paint the graph of Vector.", warningFont,
					new SolidBrush(Color.Red), 
					max_x / 2 - 160, 
					max_y / 2 - 50);
			}
		} // RenderVector()

		/// <summary>
		/// Vykresli vlastni graf (caru grafu a body).
		/// </summary>
		/// <param name="g">Povrch pro kresleni.</param>
		/// <param name="graphPen">Styl cary grafu.</param>
		private void DrawMyGraph(Graphics g, Pen graphPen)
		{
			double step = (this.maxValue - this.minValue) / (float) y_segment_count;
			System.Drawing.Point[] pointArray = new System.Drawing.Point[data.Length];
			for (int i = 0; i < this.data.Length; i++)
			{
				System.Drawing.Point point = new System.Drawing.Point((int) (VectorRendererPanel.SPACING_LEFT + i * this.segmentX), 
					(int) (VectorRendererPanel.SPACING_TOP + height - ((data[i] - this.minValue) / step) * this.segmentY));
				pointArray[i] = point;
				if (this.showGraphMarks == true)
					DrawPoint(g, point);
			}
			g.DrawLines(graphPen, pointArray);
		} // DrawMyGraph()

		/// <summary>
		/// Vykresli x-ovou osu grafu vcetne popisku.
		/// </summary>
		/// <param name="g">Kreslici povrch.</param>
		/// <param name="axisPen">Styl cary.</param>
		/// <param name="axisFont">Font popisku.</param>
		private void DrawXAxis(Graphics g, Pen axisPen, Font axisFont)
		{
			// Vodorovna osa grafu (pouze cara).
			g.DrawLine(axisPen, 
				VectorRendererPanel.SPACING_LEFT, 
				max_y - VectorRendererPanel.SPACING_BOTTOM, 
				max_x - VectorRendererPanel.SPACING_RIGHT,
				max_y - VectorRendererPanel.SPACING_BOTTOM);
			// Meritko vodorovne osy grafu.
			for (int i = 0; i < this.data.Length; i++)
			{
				if (this.showXAxesLegend == true || (this.showXAxesLegend == false && (i == 0 || i == this.data.Length - 1)))
				{
					g.DrawLine(axisPen, // svisla znacka na ose
						VectorRendererPanel.SPACING_LEFT + i * this.segmentX,
						max_y - VectorRendererPanel.SPACING_BOTTOM - VectorRendererPanel.AXIS_MARK_SIZE,
						VectorRendererPanel.SPACING_LEFT + i * this.segmentX,
						max_y - VectorRendererPanel.SPACING_BOTTOM + VectorRendererPanel.AXIS_MARK_SIZE);
					g.DrawString(i.ToString(), axisFont, // popisky
						new SolidBrush(Color.Black), 
						VectorRendererPanel.SPACING_LEFT + i * this.segmentX - 6, 
						max_y - VectorRendererPanel.SPACING_BOTTOM + 10);
				}
				if (this.showXGrid == true)
					g.DrawLine(Pens.Black, // svisla mrizka
						VectorRendererPanel.SPACING_LEFT + i * this.segmentX,
						VectorRendererPanel.SPACING_TOP,
						VectorRendererPanel.SPACING_LEFT + i * this.segmentX,
						max_y - VectorRendererPanel.SPACING_BOTTOM);
			}		
		} // DrawXAxis()

		/// <summary>
		/// Vykresli y-ovou osu grafu vcetne popisku.
		/// </summary>
		/// <param name="g">Kreslici povrch.</param>
		/// <param name="axisPen">Styl cary.</param>
		/// <param name="axisFont">Font popisku.</param>
		private void DrawYAxis(Graphics g, Pen axisPen, Font axisFont)
		{
			// Svisla osa grafu (pouze cara).
			g.DrawLine(axisPen,
				VectorRendererPanel.SPACING_LEFT, 
				max_y - VectorRendererPanel.SPACING_BOTTOM,
				VectorRendererPanel.SPACING_LEFT,
				VectorRendererPanel.SPACING_TOP);
			// Meritko svisle osy grafu.
			double step = (this.maxValue - this.minValue) / (float) y_segment_count;
			for (int i = 0; i <= y_segment_count; i++)
			{
				if (this.showYAxesLegend == true || (this.showYAxesLegend == false && (i == 0 || i == y_segment_count)))
				{
					g.DrawLine(axisPen,	// vodorovne znacky na ose
						VectorRendererPanel.SPACING_LEFT - VectorRendererPanel.AXIS_MARK_SIZE,
						VectorRendererPanel.SPACING_TOP + height - i * this.segmentY,
						VectorRendererPanel.SPACING_LEFT + VectorRendererPanel.AXIS_MARK_SIZE,
						VectorRendererPanel.SPACING_TOP + height - i * this.segmentY);
					double actual = this.minValue + i * step;
					g.DrawString(actual.ToString("f1"), axisFont, // popisky
						new SolidBrush(Color.Black), 
						VectorRendererPanel.SPACING_LEFT - 40,
						VectorRendererPanel.SPACING_TOP + height - i * this.segmentY - 7);
				}
				if (this.showYGrid == true)
					g.DrawLine(Pens.Black,	// vodorovna mrizka grafu
						VectorRendererPanel.SPACING_LEFT,
						VectorRendererPanel.SPACING_TOP + height - i * this.segmentY,
						VectorRendererPanel.SPACING_LEFT + width,
						VectorRendererPanel.SPACING_TOP + height - i * this.segmentY);
			}		
		} // DrawYAxis()

		/// <summary>
		/// Pomocna metoda pro vykresleni krizku reprezentujiciho bod v grafu.
		/// </summary>
		/// <param name="g">Graphics.</param>
		/// <param name="point">Kresleny bod.</param>
		private void DrawPoint(Graphics g, System.Drawing.Point point)
		{
			Pen pen = new Pen(new SolidBrush(marksLineColor), graphMarkLineWidth);
			g.DrawLine(pen,
				point.X - graph_mark_size,
				point.Y,
				point.X + graph_mark_size,
				point.Y);
			g.DrawLine(pen,
				point.X,
				point.Y - graph_mark_size,
				point.X,
				point.Y + graph_mark_size);
		} // DrawPoint()
		#endregion

		/// <summary>
		/// Clean up any resources being used.
		/// </summary>
		protected override void Dispose( bool disposing )
		{
			if( disposing )
			{
				if(components != null)
				{
					components.Dispose();
				}
			}
			base.Dispose( disposing );
		}

		#region Windows Form Designer generated code
		/// <summary>
		/// Required method for Designer support - do not modify
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
			this.statusBar1 = new System.Windows.Forms.StatusBar();
			this.statusBarPanel1 = new System.Windows.Forms.StatusBarPanel();
			this.mainMenu1 = new System.Windows.Forms.MainMenu();
			this.menuItem1 = new System.Windows.Forms.MenuItem();
			((System.ComponentModel.ISupportInitialize)(this.statusBarPanel1)).BeginInit();
			this.SuspendLayout();
			// 
			// statusBar1
			// 
			this.statusBar1.Font = new System.Drawing.Font("Tahoma", 8.25F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((System.Byte)(238)));
			this.statusBar1.Location = new System.Drawing.Point(0, 395);
			this.statusBar1.Name = "statusBar1";
			this.statusBar1.Panels.AddRange(new System.Windows.Forms.StatusBarPanel[] {
																						  this.statusBarPanel1});
			this.statusBar1.ShowPanels = true;
			this.statusBar1.Size = new System.Drawing.Size(624, 22);
			this.statusBar1.TabIndex = 1;
			// 
			// statusBarPanel1
			// 
			this.statusBarPanel1.AutoSize = System.Windows.Forms.StatusBarPanelAutoSize.Spring;
			this.statusBarPanel1.Width = 608;
			// 
			// mainMenu1
			// 
			this.mainMenu1.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
																					  this.menuItem1});
			this.mainMenu1.RightToLeft = System.Windows.Forms.RightToLeft.No;
			// 
			// menuItem1
			// 
			this.menuItem1.Index = 0;
			this.menuItem1.Shortcut = System.Windows.Forms.Shortcut.F10;
			this.menuItem1.Text = "Settings [F10]";
			this.menuItem1.Click += new System.EventHandler(this.menuItem1_Click);
			// 
			// VectorRendererPanel
			// 
			this.AutoScaleBaseSize = new System.Drawing.Size(5, 13);
			this.ClientSize = new System.Drawing.Size(624, 417);
			this.Controls.Add(this.statusBar1);
			this.Menu = this.mainMenu1;
			this.MinimumSize = new System.Drawing.Size(300, 200);
			this.Name = "VectorRendererPanel";
			this.Text = "VectorRenderer Window";
			((System.ComponentModel.ISupportInitialize)(this.statusBarPanel1)).EndInit();
			this.ResumeLayout(false);

		}
		#endregion

		/// <summary>
		/// Button setting click.
		/// </summary>
		/// <param name="sender">Sender.</param>
		/// <param name="e">Event parameters.</param>
		private void menuItem1_Click(object sender, System.EventArgs e)
		{
			VectorRendererSettings settingsDialog = new VectorRendererSettings(this);
			settingsDialog.ShowDialog(this);
		} // menuItem1_Click()

	} // class VectorRendererPanel
} // namespace





