using System;
using System.Globalization;
using Zcu.Mve.Core;

namespace Zcu.Mve.Numerics
{
	/// <summary>
	/// This class represents a ND vector defined by N elements.
	/// </summary>
	public class VectorND : DataObject, IVector
	{
		/// <summary>
		/// n-dimensional array of vector values.
		/// </summary>
		private double[] values;

		/// <summary>
		/// The toleration for equality operator.
		/// </summary>
		public static double Epsilon = Globals.EpsilonDoubleImplicit;

		/// <summary>
		/// Implicit constructor - creates an empty object.
		/// </summary>
		public VectorND() : this(0.0)
		{
		} // VectorND()

		/// <summary>
		/// Create a VectorND object.
		/// </summary>
		/// <param name="values">n-dimensional array of vector values.</param>
		public VectorND(params double[] values)
		{
			this.values = values;
		} // NormalND()

		/// <summary>
		/// Read data from Xml file to current structure.
		/// </summary>
		/// <param name="xmlTextReader">Stream to be read.</param>
		public override void ReadData(System.Xml.XmlTextReader xmlTextReader)
		{
			NumberFormatInfo nfi = new CultureInfo("en-US", false).NumberFormat;
			string[] dims = xmlTextReader.ReadString().Trim().Split(' ');
			this.values = new double[dims.Length];
			for (int i = 0; i < dims.Length; i++)
				this.values[i] = Double.Parse(dims[i], nfi);
		} // ReadData()

		/// <summary>
		/// Write current data to Xml file.
		/// </summary>
		/// <param name="xmlTextWriter">Stream to be write.</param>
		public override void WriteData(System.Xml.XmlTextWriter xmlTextWriter)
		{
			string pom = "";
			NumberFormatInfo nfi = new CultureInfo("en-US", false).NumberFormat;
			for (int i = 0; i < this.values.Length; i++)
				pom += this.values[i].ToString(nfi) + " ";
			pom = pom.Substring(0, pom.Length - 1);	// odstraneni prebytecne mezery na konci
			xmlTextWriter.WriteString(pom);
		} // WriteData()

		/// <summary>
		/// Deep copy of data object and its subelements.
		/// </summary>
		/// <returns>Same data, different memory space.</returns>
		public override IDataObject DeepCopy()
		{
			double[] newDims = new double[this.values.Length];
			this.values.CopyTo(newDims, 0);
			return new VectorND(newDims);
		} // DeepCopy()

		/// <summary>
		/// Check the consistency of current data structure.
		/// </summary>
		/// <returns>True if structure is consistent.</returns>
		public override bool CheckConsistence()
		{
			return true;
		} // CheckConsistence()

		/// <summary>
		/// Returns the absolute value of vector.
		/// </summary>
		/// <returns>The absolute value of vector.</returns>
		public double Abs()
		{
			double tmp = 0;
			foreach (double d in this.values)
				tmp += Math.Pow(d, 2);
			return Math.Sqrt(tmp);
		} // Abs()

		/// <summary>
		/// Get the n-dimensional array of vector values.
		/// For acces to concrete vector element use indexer.
		/// </summary>
		public double[] Values
		{
			get
			{
				return this.values;
			}
		} // Values

		/// <summary>
		/// Get/Set element of vector.
		/// </summary>
		public double this[int index]
		{
			get
			{
				if (index < this.values.Length && index >= 0)
					return this.values[index];
				else
					throw new MveException(this.GetType().Name + ": index " + index + " is out of range <0, " + (this.values.Length - 1) + ">");
			}
			set
			{
				if (index < this.values.Length && index >= 0)
					this.values[index] = value;
				else
					throw new MveException(this.GetType().Name + ": index " + index + " is out of range <0, " + (this.values.Length - 1) + ">");
			}
		} // this[int]

		/// <summary>
		/// Get number of elements of vector.
		/// </summary>
		public int Length
		{
			get
			{
				return this.values.Length;
			}
		} // Length

		// ===== Methods of the class Object ==============================================
		#region Overrided methods of the class Object

		/// <summary>
		/// Returns a String that represents the current Object.
		/// </summary>
		/// <returns>String that represents the current Object.</returns>
		public override string ToString()
		{
			string pom = "VectorND (" + this.values.Length + " dimensions): [";
			foreach (double d in values)
				pom += d + "; ";
			pom = pom.Substring(0, pom.Length - 2);  // odstraneni prebytecneho stredniku a mezery na konci
			return pom + "]";
		} // ToString()

		/// <summary>
		/// Determines whether the specified Objects are equal.
		/// </summary>
		/// <param name="a">First Object.</param>
		/// <param name="b">Second Object.</param>
		/// <returns>True if objects are equal. Otherwise, false.</returns>
		public static new bool Equals(object a, object b)
		{
			if (a.GetType() != typeof(VectorND) || b.GetType() != typeof(VectorND))
				return false;

			VectorND x = (VectorND) a;
			VectorND y = (VectorND) b;
			if (x.Length != y.Length)
				return false;

			for (int i = 0; i < x.Length; i++)
				if (Math.Abs(x[i] - y[i]) > VectorND.Epsilon)
					return false;

			return true;
		}
		
		/// <summary>
		/// Determines whether the specified Object is equal to the current Object.
		/// </summary>
		/// <param name="obj">Specified Object to compare with the current Object.</param>
		/// <returns>True if the specified Object is equal to the current Object. Otherwise, false.</returns>
		public override bool Equals(object obj)
		{
			return VectorND.Equals(this, obj);
		}
		
		/// <summary>
		/// Returns a hash code for the current Object.
		/// </summary>
		/// <returns>A hash code for the current Object.</returns>
		public override int GetHashCode()
		{
			if (this.Length == 0)
				return 0;

			if (this.Length == 1)
				return this[0].GetHashCode();

			int hashCode = this[0].GetHashCode();
			for (int i = 1; i < this.Length; i++)
				hashCode = hashCode ^ this[i].GetHashCode();

			return hashCode;
		}

		#endregion

		// ===== Conversions ==============================================================
		#region Conversion

		// ===== VectorND -> Vector ===============
		/// <summary>
		/// Explicit conversion VectorND -> Vector2D
		/// </summary>
		/// <param name="v">VectorND</param>
		/// <returns>Vector2D</returns>
		public static explicit operator Vector2D(VectorND v) 
		{
			if (v.values.Length == 1)
				return new Vector2D(v.values[0], 0);
			else
				return new Vector2D(v.values[0], v.values[1]);
		} // VectorND -> Vector2D

		/// <summary>
		/// Explicit conversion VectorND -> Vector3D
		/// </summary>
		/// <param name="v">VectorND</param>
		/// <returns>Vector3D</returns>
		public static explicit operator Vector3D(VectorND v) 
		{
			if (v.values.Length == 1)
				return new Vector3D(v.values[0], 0, 0);
			else if (v.values.Length == 2)
				return new Vector3D(v.values[0], v.values[1], 0);
			else
				return new Vector3D(v.values[0], v.values[1], v.values[2]);
		} // VectorND -> Vector3D

		#endregion

		// ===== Operators ================================================================
		#region Operators

		/// <summary>
		/// The equality operator (==) returns true if the values of its operands are equal, false otherwise.
		/// </summary>
		/// <param name="a">First operand.</param>
		/// <param name="b">Second operand.</param>
		/// <returns>True if the values of operands are equal, false otherwise.</returns>
		public static bool operator ==(VectorND a, VectorND b)
		{
			return VectorND.Equals(a, b);
		}
		
		/// <summary>
		/// The inequality operator (!=) returns false if its operands are equal, true otherwise.
		/// </summary>
		/// <param name="a">First operand.</param>
		/// <param name="b">Second operand.</param>
		/// <returns>False if the values of operands are equal, true otherwise.</returns>
		public static bool operator !=(VectorND a, VectorND b) 
		{
			return !(a == b);
		}

		#endregion

	} // class VectorND
} // namespace
