using System;
using System.Globalization;
using Zcu.Mve.Core;

namespace Zcu.Mve.Numerics
{
	/// <summary>
	/// This structure represents a 3D vector defined by three elements - X, Y and Z.
	/// </summary>
	public struct Vector3D : IVector
	{
		/// <summary>
		/// x-vector value.
		/// </summary>
		public double X;
		/// <summary>
		/// y-vector value.
		/// </summary>
		public double Y;
		/// <summary>
		/// z-vector value.
		/// </summary>
		public double Z;

		/// <summary>
		/// The toleration for equality operator.
		/// </summary>
		public static double Epsilon = Globals.EpsilonDoubleImplicit;

		/// <summary>
		/// Create a Vector3D object.
		/// </summary>
		/// <param name="vx">x-vector value.</param>
		/// <param name="vy">y-vector value.</param>
		/// <param name="vz">z-vector value.</param>
		public Vector3D(double vx, double vy, double vz)
		{
			this.X = vx;
			this.Y = vy;
			this.Z = vz;
		} // Vector3D()

		/// <summary>
		/// Read data from Xml file to current structure.
		/// </summary>
		/// <param name="xmlTextReader">Stream to be read.</param>
		public void ReadData(System.Xml.XmlTextReader xmlTextReader)
		{
			NumberFormatInfo nfi = new CultureInfo("en-US", false).NumberFormat;
			string[] split = xmlTextReader.ReadString().Trim().Split(' ');
			this.X = Double.Parse(split[0], nfi);
			this.Y = Double.Parse(split[1], nfi);
			this.Z = Double.Parse(split[2], nfi);
		} // ReadData()

		/// <summary>
		/// Write current data to Xml file.
		/// </summary>
		/// <param name="xmlTextWriter">Stream to be write.</param>
		public void WriteData(System.Xml.XmlTextWriter xmlTextWriter)
		{
			NumberFormatInfo nfi = new CultureInfo("en-US", false).NumberFormat;
			xmlTextWriter.WriteString(this.X.ToString(nfi) + " " 
				+ this.Y.ToString(nfi) + " "
				+ this.Z.ToString(nfi));
		} // WriteData()

		/// <summary>
		/// Deep copy of data object and its subelements.
		/// </summary>
		/// <returns>Same data, different memory space.</returns>
		public IDataObject DeepCopy()
		{
			return new Vector3D(this.X, this.Y, this.Z);
		} // DeepCopy()

		/// <summary>
		/// Check the consistency of current data structure.
		/// </summary>
		/// <returns>True if structure is consistent.</returns>
		public bool CheckConsistence()
		{
			return true;
		} // CheckConsistence()

		/// <summary>
		/// Returns the absolute value of vector.
		/// </summary>
		/// <returns>The absolute value of vector.</returns>
		public double Abs()
		{
			return Math.Sqrt(this.X * this.X + this.Y * this.Y + this.Z * this.Z);
		} // Abs()

		/// <summary>
		/// Get/Set element of vector.
		/// </summary>
		public double this [int index]
		{
			get
			{
				switch(index)
				{
					case 0:
						return this.X;
					case 1:
						return this.Y;
					case 2:
						return this.Z;
					default:
						throw new IndexOutOfRangeException(this.GetType().FullName + ": Dimension exceeded.");
				}
			}

			set
			{
				switch(index)
				{
					case 0:
						this.X = value;
						break;
					case 1:
						this.Y = value;
						break;
					case 2:
						this.Z = value;
						break;
					default:
						throw new IndexOutOfRangeException(this.GetType().FullName + ": Dimension exceeded.");
				}
			}
		}

		/// <summary>
		/// Get number of elements of vector.
		/// </summary>
		public int Length
		{
			get
			{
				return 3;
			}
		}

		/// <summary>
		/// Returns array of assembly name.
		/// </summary>
		public string [] LibNames
		{
			get
			{
				return new string [] {Globals.GetLibraryName(this.GetType())};
			}
		}

		// ===== Methods of the class Object ==============================================
		#region Overrided methods of the class Object

		/// <summary>
		/// Returns a String that represents the current Object.
		/// </summary>
		/// <returns>String that represents the current Object.</returns>
		override public string ToString()
		{
			return "Vector3D: [" + this.X + "; " + this.Y + "; " + this.Z + "]";
		} // ToString()

		/// <summary>
		/// Determines whether the specified Objects are equal.
		/// </summary>
		/// <param name="a">First Object.</param>
		/// <param name="b">Second Object.</param>
		/// <returns>True if objects are equal. Otherwise, false.</returns>
		public static new bool Equals(object a, object b)
		{
			if (a.GetType() != typeof(Vector3D) || b.GetType() != typeof(Vector3D))
				return false;
		      
			return Math.Abs(((Vector3D)a).X - ((Vector3D)b).X) <= Vector3D.Epsilon
				&& Math.Abs(((Vector3D)a).Y - ((Vector3D)b).Y) <= Vector3D.Epsilon
				&& Math.Abs(((Vector3D)a).Z - ((Vector3D)b).Z) <= Vector3D.Epsilon;
		}
		
		/// <summary>
		/// Determines whether the specified Object is equal to the current Object.
		/// </summary>
		/// <param name="obj">Specified Object to compare with the current Object.</param>
		/// <returns>True if the specified Object is equal to the current Object. Otherwise, false.</returns>
		public override bool Equals(object obj)
		{
			return Vector3D.Equals(this, obj);
		}
		
		/// <summary>
		/// Returns a hash code for the current Object.
		/// </summary>
		/// <returns>A hash code for the current Object.</returns>
		public override int GetHashCode()
		{
			return this.X.GetHashCode() ^ this.Y.GetHashCode() ^ this.Z.GetHashCode();
		}

		#endregion

		// ===== Conversions ==============================================================
		#region Conversion

		// ===== Vector3D -> Vector ===============
		/// <summary>
		/// Explicit conversion Vector3D -> Vector2D
		/// </summary>
		/// <param name="v">Vector3D</param>
		/// <returns>Vector2D</returns>
		public static explicit operator Vector2D(Vector3D v) 
		{
			return new Vector2D(v.X, v.Y);
		} // Vector3D -> Vector2D

		/// <summary>
		/// Implicit conversion Vector3D -> VectorND
		/// </summary>
		/// <param name="v">Vector3D</param>
		/// <returns>VectorND</returns>
		public static implicit operator VectorND(Vector3D v) 
		{
			return new VectorND(new double[] {v.X, v.Y, v.Z});
		} // Vector3D -> VectorND

		#endregion

		// ===== Operators ================================================================
		#region Operators

		/// <summary>
		/// The equality operator (==) returns true if the values of its operands are equal, false otherwise.
		/// </summary>
		/// <param name="a">First operand.</param>
		/// <param name="b">Second operand.</param>
		/// <returns>True if the values of operands are equal, false otherwise.</returns>
		public static bool operator ==(Vector3D a, Vector3D b)
		{
			return Vector3D.Equals(a, b);
		}
		
		/// <summary>
		/// The inequality operator (!=) returns false if its operands are equal, true otherwise.
		/// </summary>
		/// <param name="a">First operand.</param>
		/// <param name="b">Second operand.</param>
		/// <returns>False if the values of operands are equal, true otherwise.</returns>
		public static bool operator !=(Vector3D a, Vector3D b) 
		{
			return !(a == b);
		}

		#endregion

	} // class Vector3D
} // namespace
