using System;
using System.Globalization;
using Zcu.Mve.Core;

namespace Zcu.Mve.Numerics
{
	/// <summary>
	/// This structure represents a 2D vector defined by two elements - X and Y.
	/// </summary>
	public struct Vector2D : IVector
	{
		/// <summary>
		/// x-vector value.
		/// </summary>
		public double X;
		/// <summary>
		/// y-vector value.
		/// </summary>
		public double Y;

		/// <summary>
		/// The toleration for equality operator.
		/// </summary>
		public static double Epsilon = Globals.EpsilonDoubleImplicit;

		/// <summary>
		/// Create a Vector2D object.
		/// </summary>
		/// <param name="vx">x-vector value.</param>
		/// <param name="vy">y-vector value.</param>
		public Vector2D(double vx, double vy)
		{
			this.X = vx;
			this.Y = vy;
		} // Vector2D()

		/// <summary>
		/// Read data from Xml file to current structure.
		/// </summary>
		/// <param name="xmlTextReader">Stream to be read.</param>
		public void ReadData(System.Xml.XmlTextReader xmlTextReader)
		{
			string[] split = xmlTextReader.ReadString().Trim().Split(' ');
			this.X = Double.Parse(split[0], Globals.nfi);
			this.Y = Double.Parse(split[1], Globals.nfi);
		} // ReadData()

		/// <summary>
		/// Write current data to Xml file.
		/// </summary>
		/// <param name="xmlTextWriter">Stream to be write.</param>
		public void WriteData(System.Xml.XmlTextWriter xmlTextWriter)
		{
			NumberFormatInfo nfi = new CultureInfo("en-US", false).NumberFormat;
			xmlTextWriter.WriteString(this.X.ToString(nfi) + " " + this.Y.ToString(nfi));
		} // WriteData()

		/// <summary>
		/// Deep copy of data object and its subelements.
		/// </summary>
		/// <returns>Same data, different memory space.</returns>
		public IDataObject DeepCopy()
		{
			return new Vector2D(this.X, this.Y);
		} // DeepCopy()

		/// <summary>
		/// Check the consistency of current data structure.
		/// </summary>
		/// <returns>True if structure is consistent.</returns>
		public bool CheckConsistence()
		{
			return true;
		} // CheckConsistence()

		/// <summary>
		/// Returns the absolute value of vector.
		/// </summary>
		/// <returns>The absolute value of vector.</returns>
		public double Abs()
		{
			return Math.Sqrt(this.X * this.X + this.Y * this.Y);
		} // Abs()

		/// <summary>
		/// Get/Set element of vector.
		/// </summary>
		public double this [int index]
		{
			get
			{
				switch(index)
				{
					case 0:
						return this.X;
					case 1:
						return this.Y;
					default:
						throw new IndexOutOfRangeException(this.GetType().FullName + ": Dimension exceeded.");
				}
			}

			set
			{
				switch(index)
				{
					case 0:
						this.X = value;
						break;
					case 1:
						this.Y = value;
						break;
					default:
						throw new IndexOutOfRangeException(this.GetType().FullName + ": Dimension exceeded.");
				}
			}
		}

		/// <summary>
		/// Get number of elements of vector.
		/// </summary>
		public int Length
		{
			get
			{
				return 2;
			}
		}

		/// <summary>
		/// Returns array of assembly name.
		/// </summary>
		public string [] LibNames
		{
			get
			{
				return new string [] {Globals.GetLibraryName(this.GetType())};
			}
		}

		// ===== Methods of the class Object ==============================================
		#region Overrided methods of the class Object

		/// <summary>
		/// Returns a String that represents the current Object.
		/// </summary>
		/// <returns>String that represents the current Object.</returns>
		override public string ToString()
		{
			return "Vector2D: [" + this.X + "; " + this.Y + "]";
		} // ToString()

		/// <summary>
		/// Determines whether the specified Objects are equal.
		/// </summary>
		/// <param name="a">First Object.</param>
		/// <param name="b">Second Object.</param>
		/// <returns>True if objects are equal. Otherwise, false.</returns>
		public static new bool Equals(object a, object b)
		{
			if (a.GetType() != typeof(Vector2D) || b.GetType() != typeof(Vector2D))
				return false;
      
			return Math.Abs(((Vector2D)a).X - ((Vector2D)b).X) <= Vector2D.Epsilon
				&& Math.Abs(((Vector2D)a).Y - ((Vector2D)b).Y) <= Vector2D.Epsilon;
		}

		/// <summary>
		/// Determines whether the specified Object is equal to the current Object.
		/// </summary>
		/// <param name="obj">Specified Object to compare with the current Object.</param>
		/// <returns>True if the specified Object is equal to the current Object. Otherwise, false.</returns>
		public override bool Equals(object obj)
		{
			return Vector2D.Equals(this, obj);
		}

		/// <summary>
		/// Returns a hash code for the current Object.
		/// </summary>
		/// <returns>A hash code for the current Object.</returns>
		public override int GetHashCode()
		{
			return this.X.GetHashCode() ^ this.Y.GetHashCode();
		}

		#endregion

		// ===== Conversions ==============================================================
		#region Conversion

		// ===== Vector2D -> Vector ===============
		/// <summary>
		/// Implicit conversion Vector2D -> Vector3D
		/// </summary>
		/// <param name="v">Vector2D</param>
		/// <returns>Vector3D</returns>
		public static implicit operator Vector3D(Vector2D v) 
		{
			return new Vector3D(v.X, v.Y, 0);
		} // Vector2D -> Vector3D

		/// <summary>
		/// Implicit conversion Vector2D -> VectorND
		/// </summary>
		/// <param name="v">Vector2D</param>
		/// <returns>VectorND</returns>
		public static implicit operator VectorND(Vector2D v) 
		{
			return new VectorND(new double[] {v.X, v.Y});
		} // Vector2D -> VectorND

		#endregion

		// ===== Operators ================================================================
		#region Operators

		/// <summary>
		/// The equality operator (==) returns true if the values of its operands are equal, false otherwise.
		/// </summary>
		/// <param name="a">First operand.</param>
		/// <param name="b">Second operand.</param>
		/// <returns>True if the values of operands are equal, false otherwise.</returns>
		public static bool operator ==(Vector2D a, Vector2D b)
		{
			return Vector2D.Equals(a, b);
		}

		/// <summary>
		/// The inequality operator (!=) returns false if its operands are equal, true otherwise.
		/// </summary>
		/// <param name="a">First operand.</param>
		/// <param name="b">Second operand.</param>
		/// <returns>False if the values of operands are equal, true otherwise.</returns>
		public static bool operator !=(Vector2D a, Vector2D b) 
		{
			return !(a == b);
		}

		#endregion

	} // class Vector2D
} // namespace
