using System;
using System.Globalization;
using Zcu.Mve.Core;

namespace Zcu.Mve.Numerics
{
	/// <summary>
	/// This structure represents a scalar value.
	/// </summary>
	public struct Scalar : IDataObject
	{
		/// <summary>
		/// Scalar value.
		/// </summary>
		public double Value;

		/// <summary>
		/// The toleration for equality operator.
		/// </summary>
		public static double Epsilon = Globals.EpsilonDoubleImplicit;

		/// <summary>
		/// Create a Scalar object.
		/// </summary>
		/// <param name="val">Scalar value.</param>
		public Scalar(double val)
		{
			this.Value = val;
		} // Scalar()

		/// <summary>
		/// Read data from Xml file to current structure.
		/// </summary>
		/// <param name="xmlTextReader">Stream to be read.</param>
		public void ReadData(System.Xml.XmlTextReader xmlTextReader)
		{
			NumberFormatInfo nfi = new CultureInfo("en-US", false).NumberFormat;
			string tmp = xmlTextReader.ReadString().Trim();
			this.Value = Double.Parse(tmp, nfi);
		} // ReadData()

		/// <summary>
		/// Write current data to Xml file.
		/// </summary>
		/// <param name="xmlTextWriter">Stream to be write.</param>
		public void WriteData(System.Xml.XmlTextWriter xmlTextWriter)
		{
			NumberFormatInfo nfi = new CultureInfo("en-US", false).NumberFormat;
			xmlTextWriter.WriteString(this.Value.ToString(nfi));
		} // WriteData()

		/// <summary>
		/// Deep copy of data object and its subelements.
		/// </summary>
		/// <returns>Same data, different memory space.</returns>
		public IDataObject DeepCopy()
		{
			return new Scalar(this.Value);
		} // DeepCopy()

		/// <summary>
		/// Check the consistency of current data structure.
		/// </summary>
		/// <returns>True if structure is consistent.</returns>
		public bool CheckConsistence()
		{
			return true;
		} // CheckConsistence()

		/// <summary>
		/// Returns array of assembly name.
		/// </summary>
		public string [] LibNames
		{
			get
			{
				return new string [] {Globals.GetLibraryName(this.GetType())};
			}
		}

		// ===== Methods of the class Object ==============================================
		#region Overrided methods of the class Object

		/// <summary>
		/// Returns a String that represents the current Object.
		/// </summary>
		/// <returns>String that represents the current Object.</returns>
		public override string ToString()
		{
			return "Scalar: " + this.Value;
		} // ToString()

		/// <summary>
		/// Determines whether the specified Objects are equal.
		/// </summary>
		/// <param name="a">First Object.</param>
		/// <param name="b">Second Object.</param>
		/// <returns>True if objects are equal. Otherwise, false.</returns>
		public static new bool Equals(object a, object b)
		{
			if (a.GetType() != typeof(Scalar) || b.GetType() != typeof(Scalar))
				return false;
      
			return Math.Abs(((Scalar)a).Value - ((Scalar)b).Value) <= Scalar.Epsilon;
		}

		/// <summary>
		/// Determines whether the specified Object is equal to the current Object.
		/// </summary>
		/// <param name="obj">Specified Object to compare with the current Object.</param>
		/// <returns>True if the specified Object is equal to the current Object. Otherwise, false.</returns>
		public override bool Equals(object obj)
		{
			return Scalar.Equals(this, obj);
		}

		/// <summary>
		/// Returns a hash code for the current Object.
		/// </summary>
		/// <returns>A hash code for the current Object.</returns>
		public override int GetHashCode()
		{
			return this.Value.GetHashCode();
		}

		#endregion

		// ===== Conversions ==============================================================
		#region Conversions

		// ===== Scalar <-> double ================
		/// <summary>
		/// Implicit conversion Scalar -> double.
		/// </summary>
		/// <param name="s">Scalar.</param>
		/// <returns>Double.</returns>
		public static implicit operator double(Scalar s) 
		{
			return s.Value;
		} // Scalar -> double

		/// <summary>
		/// Implicit conversion double -> Scalar
		/// </summary>
		/// <param name="d">Double.</param>
		/// <returns>Scalar.</returns>
		public static implicit operator Scalar(double d)
		{
			return new Scalar(d);
		} // double -> Scalar

		// ===== Scalar <-> float =================
		/// <summary>
		/// Implicit conversion Scalar -> float
		/// </summary>
		/// <param name="s">Scalar.</param>
		/// <returns>Float.</returns>
		public static implicit operator float(Scalar s) 
		{
			return (float) s.Value;
		} // Scalar -> float

		/// <summary>
		/// Implicit conversion float -> Scalar
		/// </summary>
		/// <param name="f">Float.</param>
		/// <returns>Scalar.</returns>
		public static implicit operator Scalar(float f)
		{
			return new Scalar(f);
		} // float -> Scalar

		#endregion

		// ===== Operators ================================================================
		#region Operators

		/// <summary>
		/// The equality operator (==) returns true if the values of its operands are equal, false otherwise.
		/// </summary>
		/// <param name="a">First operand.</param>
		/// <param name="b">Second operand.</param>
		/// <returns>True if the values of operands are equal, false otherwise.</returns>
		public static bool operator ==(Scalar a, Scalar b)
		{
			return Scalar.Equals(a, b);
		}

		/// <summary>
		/// The inequality operator (!=) returns false if its operands are equal, true otherwise.
		/// </summary>
		/// <param name="a">First operand.</param>
		/// <param name="b">Second operand.</param>
		/// <returns>False if the values of operands are equal, true otherwise.</returns>
		public static bool operator !=(Scalar a, Scalar b) 
		{
			return !(a == b);
		}

		#endregion
    
	} // class Scalar
} // namespace
